require "prefabutil"


local anims =
{
    { threshold = 0, anim = "broken" },
    { threshold = 0.4, anim = "onequarter" },
    { threshold = 0.5, anim = "half" },
    { threshold = 0.99, anim = "threequarter" },
    { threshold = 1, anim = { "fullA", "fullB", "fullC" } },
}

local function resolveanimtoplay(inst, percent)
    for i, v in ipairs(anims) do
        if percent <= v.threshold then
            if type(v.anim) == "table" then
                local x, y, z = inst.Transform:GetWorldPosition()
                local x = math.floor(x)
                local z = math.floor(z)
                local q1 = #v.anim + 1
                local q2 = #v.anim + 4
                local t = ( ((x%q1)*(x+3)%q2) + ((z%q1)*(z+3)%q2) )% #v.anim + 1
                return v.anim[t]
            else
                return v.anim
            end
        end
    end
end


function MakeMechType(data)

	local assets =
	{
		Asset("ANIM", "anim/wall.zip"),
		Asset("ANIM", "anim/wall_".. data.name..".zip"),
		Asset("ATLAS", "images/inventoryimages/mech_"..data.name.."_item.xml"),
	}

	local prefabs =
	{
		"collapse_small",
	}

		
	local function makeobstacle(inst)	
		inst.Physics:SetCollisionGroup(COLLISION.OBSTACLES)	
	    inst.Physics:ClearCollisionMask()
		inst.Physics:SetMass(0)
		inst.Physics:CollidesWith(COLLISION.ITEMS)
		inst.Physics:CollidesWith(COLLISION.CHARACTERS)
		inst.Physics:SetActive(true)
	    local ground = GetWorld()
	    if ground then
	    	local pt = Point(inst.Transform:GetWorldPosition())
	    	ground.Pathfinder:AddWall(pt.x, pt.y, pt.z)
	    end
	end

	local function clearobstacle(inst)
	    inst:DoTaskInTime(2*FRAMES, function() inst.Physics:SetActive(false) end)

	    local ground = GetWorld()
	    if ground then
	    	local pt = Point(inst.Transform:GetWorldPosition())
	    	ground.Pathfinder:RemoveWall(pt.x, pt.y, pt.z)
	    end
	end
		
	local function closewallremote(inst)
	local var = inst.components.health:GetPercent()
		if var <= 0 then
			if data.destroysound then
				inst.SoundEmitter:PlaySound(data.destroysound)
			end
			inst.components.wallgates.isopen = true
			inst.AnimState:PushAnimation("broken")
		elseif var <= .4 then
			inst.components.wallgates.isopen = false
			inst.AnimState:PlayAnimation("onequarter")
			if data.buildsound then
				inst.SoundEmitter:PlaySound(data.buildsound)
			end
			makeobstacle(inst)
		elseif var <= .5 then
			inst.components.wallgates.isopen = false
			inst.AnimState:PlayAnimation("half")
			if data.buildsound then
				inst.SoundEmitter:PlaySound(data.buildsound)
			end
			makeobstacle(inst)
		elseif var <= .9 then
			inst.components.wallgates.isopen = false
			inst.AnimState:PlayAnimation("threequarter")
			if data.buildsound then
				inst.SoundEmitter:PlaySound(data.buildsound)
			end
			makeobstacle(inst)
		else
			inst.AnimState:PlayAnimation(resolveanimtoplay(inst, inst.components.health:GetPercent()))
			if data.buildsound then
				inst.SoundEmitter:PlaySound(data.buildsound)
			end
			inst.components.wallgates.isopen = false
			makeobstacle(inst)
		end
	end
	
	local function closewall(inst)
		local x,y,z = inst.Transform:GetWorldPosition()
		local nearbywallgates = TheSim:FindEntities(x,y,z, 2, {"wallgate"} )
		for i = 2, #nearbywallgates do
			if nearbywallgates[i].components.wallgates.isopen == true and not nearbywallgates[i]:HasTag("broken") then
				closewallremote(nearbywallgates[i])
			end
		end
		
		local var = inst.components.health:GetPercent()
		if var <= 0 then
			if data.destroysound then
				inst.SoundEmitter:PlaySound(data.destroysound)
			end	
			inst.components.wallgates.isopen = true
			inst.AnimState:PushAnimation("broken")
		elseif var <= .4 then
			inst.components.wallgates.isopen = false
			inst.AnimState:PlayAnimation("onequarter")
			if data.buildsound then
				inst.SoundEmitter:PlaySound(data.buildsound)
			end
			makeobstacle(inst)
		elseif var <= .5 then
			inst.components.wallgates.isopen = false
			inst.AnimState:PlayAnimation("half")
			if data.buildsound then
				inst.SoundEmitter:PlaySound(data.buildsound)
			end
			makeobstacle(inst)
		elseif var <= .9 then
			inst.components.wallgates.isopen = false
			inst.AnimState:PlayAnimation("threequarter")
			if data.buildsound then
				inst.SoundEmitter:PlaySound(data.buildsound)
			end
			makeobstacle(inst)
		else
			inst.AnimState:PlayAnimation(resolveanimtoplay(inst, inst.components.health:GetPercent()))
			if data.buildsound then
				inst.SoundEmitter:PlaySound(data.buildsound)
			end
			inst.components.wallgates.isopen = false
			makeobstacle(inst)
		end
	end
	
	local function quantizeposition(pt)
		local retval = Vector3(math.floor(pt.x)+.5, 0, math.floor(pt.z)+.5)
		return retval
	end 

	local function ondeploywall(inst, pt, deployer)
	local mech = SpawnPrefab("mech_"..data.name) 
        if mech ~= nil then 
			pt = quantizeposition(pt)
            mech.Physics:SetCollides(false)
            mech.Physics:Teleport(pt.x, pt.y, pt.z) 
            mech.Physics:SetCollides(true)
            inst.components.stackable:Get():Remove()
			local ground = GetWorld()
		    if ground then
		    	ground.Pathfinder:AddWall(pt.x, pt.y, pt.z)
		    end
        end		
	end
	
	local function openwallremote(inst)
	    inst.components.wallgates.isopen = true
	    if data.destroysound then
			inst.SoundEmitter:PlaySound(data.destroysound)
		end	
	    inst.AnimState:PlayAnimation("broken")
	    clearobstacle(inst)
	end
	
	local function openwall(inst)
		local x,y,z = inst.Transform:GetWorldPosition()
		local nearbywallgates = TheSim:FindEntities(x,y,z, 2, {"wallgate"} )
		for i = 2, #nearbywallgates do
			if nearbywallgates[i].components.wallgates.isopen == false and not nearbywallgates[i]:HasTag("broken") then
				openwallremote(nearbywallgates[i])
			end
		end
		
		inst.components.wallgates.isopen = true
	    if data.destroysound then
			inst.SoundEmitter:PlaySound(data.destroysound)
		end		
	    inst.AnimState:PlayAnimation("broken")
	    clearobstacle(inst)
	end
	
	local function onhammered(inst, worker)
		if data.maxloots and data.loot then
			local num_loots = math.max(1, math.floor(data.maxloots*inst.components.health:GetPercent()))
			for k = 1, num_loots do
				inst.components.lootdropper:SpawnLootPrefab(data.loot)
			end
		end		
		
		SpawnPrefab("collapse_small").Transform:SetPosition(inst.Transform:GetWorldPosition())
		
		if data.destroysound then
			inst.SoundEmitter:PlaySound(data.destroysound)		
		end
		local modname = KnownModIndex:GetModActualName("Wall Gates")
		local loot_ver = GetModConfigData("Wall Gates Recipe", modname)
		if loot_ver == "gears" then
			SpawnPrefab("gears").Transform:SetPosition(inst.Transform:GetWorldPosition())
			SpawnPrefab("goldnugget").Transform:SetPosition(inst.Transform:GetWorldPosition())
		elseif loot_ver == "transistor" then
			SpawnPrefab("goldnugget").Transform:SetPosition(inst.Transform:GetWorldPosition())
			SpawnPrefab("transistor").Transform:SetPosition(inst.Transform:GetWorldPosition())
		elseif loot_ver == "gold" then
			SpawnPrefab("goldnugget").Transform:SetPosition(inst.Transform:GetWorldPosition())
			SpawnPrefab("goldnugget").Transform:SetPosition(inst.Transform:GetWorldPosition())
		end
		inst:Remove()
	end

	local function ongusthammerfn(inst)
	    inst.components.health:DoDelta(-data.windblown_damage, false, "wind")
	end

	local function test_wall(inst, pt)
		local map = GetWorld().Map
		local tiletype = GetGroundTypeAtPosition(pt)
		local ground_OK = tiletype ~= GROUND.IMPASSABLE and not map:IsWater(tiletype)
			
		if ground_OK then
			local ents = TheSim:FindEntities(pt.x,pt.y,pt.z, 2, nil, {"NOBLOCK", "player", "FX", "INLIMBO", "DECOR"}) -- or we could include a flag to the search?

			for k, v in pairs(ents) do
				if v ~= inst and v.entity:IsValid() and v.entity:IsVisible() and not v.components.placer and v.parent == nil then
					local dsq = distsq( Vector3(v.Transform:GetWorldPosition()), pt)
					if v:HasTag("wall") then
						if dsq < .1 then return false end
					else
						if  dsq< 1 then return false end
					end
				end
			end

			local playerPos = GetPlayer():GetPosition()
			local xDiff = playerPos.x - pt.x 
			local zDiff = playerPos.z - pt.z 
			local dsq = xDiff * xDiff + zDiff * zDiff
			if dsq < .5 then 
				return false 
			end 

			return true

		end
		return false
		
	end

	local function onhealthchange(inst, old_percent, new_percent)
		if old_percent <= 0 and new_percent > 0 then inst:RemoveTag("broken") makeobstacle(inst) end
		if old_percent > 0 and new_percent <= 0 then inst:AddTag("broken") clearobstacle(inst) end

		local anim_to_play = resolveanimtoplay(inst, new_percent)
		if new_percent > 0 and inst.components.wallgates:IsOpen() then
			inst.AnimState:PlayAnimation("broken")
		elseif new_percent > 0 and not inst.components.wallgates:IsOpen() then
			inst.AnimState:PlayAnimation(anim_to_play.."_hit")      
			inst.AnimState:PushAnimation(anim_to_play, false)       
		else
			inst.AnimState:PlayAnimation(anim_to_play)      
		end
	end
	
	local function itemfn(Sim)
		local inst = CreateEntity()
		inst:AddTag("wallbuilder")
		inst:AddTag("wallgateitem")
				
		inst.entity:AddTransform()
		inst.entity:AddAnimState()
		MakeInventoryPhysics(inst)
	    
		inst.AnimState:SetBank("wall")
		inst.AnimState:SetBuild("wall_"..data.name)
		inst.AnimState:PlayAnimation("idle")

		if data.name == "wood" or data.name == "hay" then
			MakeInventoryFloatable(inst, "idle_water", "idle")
		end

		inst:AddComponent("stackable")
		inst.components.stackable.maxsize = TUNING.STACK_SIZE_MEDITEM

		inst:AddComponent("inspectable")
		inst:AddComponent("inventoryitem")
		inst.components.inventoryitem.atlasname = "images/inventoryimages/mech_"..data.name.."_item.xml"
		
		if data.flammable then
			MakeSmallBurnable(inst, TUNING.MED_BURNTIME)
			MakeSmallPropagator(inst)
			
			inst:AddComponent("fuel")
			inst.components.fuel.fuelvalue = TUNING.SMALL_FUEL

			inst:AddComponent("appeasement")
    		inst.components.appeasement.appeasementvalue = TUNING.WRATH_SMALL

			inst.components.burnable:MakeDragonflyBait(3)
		end
		
		inst:AddComponent("deployable")
		inst.components.deployable.ondeploy = ondeploywall
		inst.components.deployable.test = test_wall
		inst.components.deployable.min_spacing = 0
		inst.components.deployable.placer = "wall_"..data.name.."_placer"
		inst.components.deployable:SetQuantizeFunction(quantizeposition)
		inst.components.deployable.deploydistance = 1.5
		
		return inst
	end

    local function onhit(inst)
        if data.destroysound then
            inst.SoundEmitter:PlaySound(data.destroysound)
        end

        local healthpercent = inst.components.health:GetPercent()
        local anim_to_play = resolveanimtoplay(inst, healthpercent)
		if healthpercent > 0 and inst.components.wallgates:IsOpen() then
			inst.AnimState:PlayAnimation("broken")
        elseif healthpercent > 0 then
            inst.AnimState:PlayAnimation(anim_to_play.."_hit")
            inst.AnimState:PushAnimation(anim_to_play, false)
        end
    end

    local function onrepaired(inst)
        if data.buildsound then
            inst.SoundEmitter:PlaySound(data.buildsound)
        end
        openwall(inst)
    end
	
	local function onload(inst)
		if inst.components.health:IsDead() then
			inst:AddTag("broken")
			clearobstacle(inst)
		end
	end

	local function onremoveentity(inst)
		clearobstacle(inst)
	end

	local function fn(Sim)
		local inst = CreateEntity()
		local trans = inst.entity:AddTransform()
		local anim = inst.entity:AddAnimState()
		inst.entity:AddSoundEmitter()
		trans:SetEightFaced()
		
		inst:AddTag("wall")
		inst:AddTag("wallgate")
		
		MakeObstaclePhysics(inst, .5)    
		inst.entity:SetCanSleep(false)
		anim:SetBank("wall")
		anim:SetBuild("wall_"..data.name)
	    anim:PlayAnimation("half", false)
	    
		inst:AddComponent("inspectable")
		inst:AddComponent("lootdropper")
		
		for k,v in ipairs(data.tags) do
		    inst:AddTag(v)
		end
				
		inst:AddComponent("repairable")
        if data.name == "ruins" then
		    inst.components.repairable.repairmaterial = "thulecite"
        else
		    inst.components.repairable.repairmaterial = data.name
        end
		inst.components.repairable.onrepaired = onrepaired
		
		inst:AddComponent("combat")
		inst.components.combat.onhitfn = onhit
		
		inst:AddComponent("health")
		inst.components.health:SetMaxHealth(data.maxhealth)
		inst.components.health.currenthealth = data.maxhealth / 2
		inst.components.health.ondelta = onhealthchange
		inst.components.health.nofadeout = true
		inst.components.health.canheal = false
		inst:AddTag("noauradamage")
		
		inst:AddComponent("wallgates")
		inst.components.wallgates.openwallfn = openwall
        inst.components.wallgates.closewallfn = closewall
		
		if data.flammable then
			MakeMediumBurnable(inst)
			MakeLargePropagator(inst)
			inst.components.burnable.flammability = .5
			inst.components.burnable.canlight = false
			
			if data.name == "wood" then
				inst.components.propagator.flashpoint = 30+math.random()*10			
			end
		else
			inst.components.health.fire_damage_scale = 0
		end

		if data.buildsound then
			inst.SoundEmitter:PlaySound(data.buildsound)		
		end
		
		inst:AddComponent("workable")
		inst.components.workable:SetWorkAction(ACTIONS.HAMMER)
		inst.components.workable:SetWorkLeft(3)
		inst.components.workable:SetOnFinishCallback(onhammered)
		inst.components.workable:SetOnWorkCallback(onhit) 

		if data.windblown_speed and data.windblown_fall_chance and data.windblown_damage then
		    inst:AddComponent("blowinwindgust")
		    inst.components.blowinwindgust:SetWindSpeedThreshold(data.windblown_speed)
		    inst.components.blowinwindgust:SetDestroyChance(data.windblown_fall_chance)
		    inst.components.blowinwindgust:SetDestroyFn(ongusthammerfn)
		    inst.components.blowinwindgust:Start()
		end	
		
	    inst.OnLoad = onload
	    inst.OnRemoveEntity = onremoveentity
		
		MakeSnowCovered(inst)
		
		return inst
	end


	return Prefab( "common/mech_"..data.name, fn, assets, prefabs),
		   Prefab( "common/mech_"..data.name.."_item", itemfn, assets, {"wall_"..data.name, "mech_"..data.name.."_placer", "collapse_small"}),
		   MakePlacer("common/mech_"..data.name.."_placer", "wall", "wall_"..data.name, "half", false, false, true, nil, nil, nil, "eight") 
	end



local mechprefabs = {}

local mechdata = {
			{name = "stone", tags={"stone"}, loot = "rocks", maxloots = 2, maxhealth=TUNING.STONEWALL_HEALTH, buildsound="dontstarve/common/place_structure_stone", destroysound="dontstarve/common/destroy_stone"},
			{name = "wood", tags={"wood"}, loot = "log", maxloots = 2, maxhealth=TUNING.WOODWALL_HEALTH, flammable = true, buildsound="dontstarve/common/place_structure_wood", destroysound="dontstarve/common/destroy_wood", windblown_speed=TUNING.WALLWOOD_WINDBLOWN_SPEED, windblown_fall_chance=TUNING.WALLWOOD_WINDBLOWN_DAMAGE_CHANCE, windblown_damage=TUNING.WALLWOOD_WINDBLOWN_DAMAGE},
			{name = "hay", tags={"grass"}, loot = "cutgrass", maxloots = 2, maxhealth=TUNING.HAYWALL_HEALTH, flammable = true, buildsound="dontstarve/common/place_structure_straw", destroysound="dontstarve/common/destroy_straw", windblown_speed=TUNING.WALLHAY_WINDBLOWN_SPEED, windblown_fall_chance=TUNING.WALLHAY_WINDBLOWN_DAMAGE_CHANCE, windblown_damage=TUNING.WALLHAY_WINDBLOWN_DAMAGE},
			{name = "ruins", tags={"stone", "ruins"}, loot = "thulecite_pieces", maxloots = 2, maxhealth=TUNING.RUINSWALL_HEALTH, buildsound="dontstarve/common/place_structure_stone", destroysound="dontstarve/common/destroy_stone"},
			{name = "limestone", tags={"stone"}, loot = "coral", maxloots = 2, maxhealth=TUNING.LIMESTONEWALL_HEALTH, buildsound="dontstarve/common/place_structure_stone", destroysound="dontstarve/common/destroy_stone"},
        }


for k,v in pairs(mechdata) do
	local mech, item, placer = MakeMechType(v)
	table.insert(mechprefabs, mech)
	table.insert(mechprefabs, item)
	table.insert(mechprefabs, placer)
end


return unpack(mechprefabs) 
